import { BlockPermutation, EntityEquippableComponent, EquipmentSlot, GameMode, ItemStack, Player } from '@minecraft/server';
import { MinecraftBlockItemTypes } from '../item/MinecraftBlockItemTypes';

export class Helper {
  /**
   * Compare 2 Vector3 (x,y,z).
   * @param {import("@minecraft/server").Vector3} v1 The first Vector3
   * @param {import("@minecraft/server").Vector3} v2 The second Vector3
   * @returns {boolean} True if are the same Vector3.
   */
  static compareVector3(v1, v2) {
    return v1.x === v2.x && v1.y === v2.y && v1.z === v2.z;
  }

  /**
   * Stringify a Vector3.
   * @param {import("@minecraft/server").Vector3} v The Vector3 to be stringified.
   * @param {boolean} includeCommas Keep the commas in the stringified version.
   * @returns {string} Stringified Vector3.
   */
  static stringifyVector3(v, includeCommas = false) {
    const c = includeCommas ? ',' : '';
    return `${v.x}${c} ${v.y}${c} ${v.z}`;
  }

  /**
   * Checks if an Item is also a Block.
   * @param {ItemStack} item The item to be validate.
   * @returns {boolean} True if the item is a block.
   */
  static isItemOrBlockItem(item) {
    if (MinecraftBlockItemTypes.includes(item.typeId)) {
      return true;
    }

    try {
      BlockPermutation.resolve(item.typeId);
      return false;
    } catch (_err) {
      return true;
    }
  }

  /**
   * Automatically decreate the ItemStack from the player's mainhand.
   * @param {Player} player The interactive player.
   * @param {number} amount How much the item should decrease Default value: 1
   */
  static decreaseMainhandItemStack(player, amount = 1) {
    const equipments = player.getComponent(EntityEquippableComponent.componentId);
    const mainhandItem = equipments.getEquipment(EquipmentSlot.Mainhand);
    
    const updatedItemAmount = mainhandItem.amount - amount;

    if (player.matches({ gameMode: GameMode.creative })) return;

    equipments.setEquipment(EquipmentSlot.Mainhand,
      updatedItemAmount !== 0 ?
        new ItemStack(mainhandItem.typeId, updatedItemAmount) :
        null
    );
  }
}
